-- Start Code --

-- TODO: Auto-detection of iTach, allowing iTach to send out multiple ports at a time...  Maybe v.2 driver...
-- 3/31/10 -- Fixed Defect #26774 - Needs auto-reconnect on disconnect from device, including any serial ports that need re-connecting...

-- print(string.format("%.2f", collectgarbage("count")) .. " Kilobytes RAM used.")
-- TODO: Figure out if we need to replace tonumber with internationalized version...

gGCBinding = 6001
gGCControlPort = 4998

gDirectorBinding = 6002
gDirectorPort = 5020

FIRST_SERIAL_IP_PORT = 4999

-- Control4 binding types for equivalent module types on GC
BINDING_TYPES = {}
BINDING_TYPES["IR"] = {"IR_OUT", "CONTACT_SENSOR"}
BINDING_TYPES["IR_BLASTER"] = {"IR_OUT", "CONTACT_SENSOR"}
BINDING_TYPES["FLEX_IR"] = {"IR_OUT"}
BINDING_TYPES["FLEX_IR_BLASTER"] = {"IR_OUT"}
BINDING_TYPES["IRTRIPORT"] = {"IR_OUT"}
BINDING_TYPES["IRTRIPORT_BLASTER"] = {"IR_OUT"}
BINDING_TYPES["SERIAL"] = {"RS_232"}
BINDING_TYPES["FLEXSERIAL"] = {"RS_232", "RS_485"}
BINDING_TYPES["RELAY"] = {"RELAY"}
BINDING_TYPES["RELAYSENSOR"] = {"RELAY", "CONTACT_SENSOR"}
BINDING_TYPES["SENSOR"] = {"CONTACT_SENSOR"}
BINDING_TYPES["SENSOR_NOTIFY"] = {"CONTACT_SENSOR"}

-- Binding names per module type on GC
TYPE_NAME = {}
TYPE_NAME["IR"] = "IR Output/Sensor Input "
TYPE_NAME["IR_BLASTER"] = "IR Blaster "
TYPE_NAME["FLEX_IR"] = "IR Output "
TYPE_NAME["FLEX_IR_BLASTER"] = "IR Blaster "
TYPE_NAME["IRTRIPORT"] = "IR Triport "
TYPE_NAME["IRTRIPORT_BLASTER"] = "IR Triport Blaster "
TYPE_NAME["SERIAL"] = "Serial Port "
TYPE_NAME["FLEXSERIAL"] = "Serial Port "
TYPE_NAME["RELAY"] = "Relay Port "
TYPE_NAME["RELAYSENSOR"] = "Relay/Sensor"
TYPE_NAME["SENSOR"] = "Sensor "
TYPE_NAME["SENSOR_NOTIFY"] = "Sensor/Notify "

-- Blink states on GC
BLINK = {}
BLINK.On = 1
BLINK.Off = 0
BLINK[true] = 1
BLINK[false] = 0

-- Relay states on GC, from GC to C4
RELAYSTATE = {}
RELAYSTATE["OPENED"] = "0"
RELAYSTATE["CLOSED"] = "1"
RELAYSTATE["0"] = "OPENED"
RELAYSTATE["1"] = "CLOSED"

-- Flow control, Parity parameters from C4 to GC
FLOW = {}
FLOW["none"] = "FLOW_NONE"
FLOW["hardware"] = "FLOW_HARDWARE"
PARITY = {}
PARITY["none"] = "PARITY_NO"
PARITY["odd"] = "PARITY_ODD"
PARITY["even"] = "PARITY_EVEN"

IRMODE = {}
IRMODE["With Carrier"] = "IR"
IRMODE["No Carrier"] = "IR_NOCARRIER"
IRMODE["GC BL2 Blaster"] = "BL2_BLASTER"

STARS = {}
STARS = {'/', '-', '\\', '|'}
gCurStar = 1

Version = ""

function OnDriverDestroyed()
    -- Kill all timers in the system...
    if (g_dbgTimer ~= nil) then C4:KillTimer(g_dbgTimer) end
    if (gCmdTimer ~= nil) then gCmdTimer = C4:KillTimer(gCmdTimer) end
    if (gReconnectTimer ~= nil) then gReconnectTimer = C4:KillTimer(gReconnectTimer) end
    if (gIRDelayTimer ~= nil) then gIRDelayTimer = C4:KillTimer(gIRDelayTimer) end
    if (gSerialSettingsTimer ~= nil) then gSerialSettingsTimer = C4:KillTimer(gSerialSettingsTimer) end
    if (gTriggerTimer ~= nil) then gTriggerTimer = C4:KillTimer(gTriggerTimer) end
    if (gPollTimer ~= nil) then gPollTimer = C4:KillTimer(gPollTimer) end
    if (gStatePollTimer ~= nil) then gStatePollTimer = C4:KillTimer(gStatePollTimer) end
    if (g_DeviceIterationTimer ~= nil) then C4:KillTimer(g_DeviceIterationTimer) end
    if (g_InIRTimer ~= nil) then C4:KillTimer(g_InIRTimer) end
end


function OnPropertyChanged(strProperty)
    local prop = Properties[strProperty]
    
    if (strProperty == "Debug Mode") then
        if (g_dbgTimer > 0) then g_dbgTimer = C4:KillTimer(g_dbgTimer) end
        g_dbgprint, g_dbglog = (prop:find("Print") ~= nil), (prop:find("Log") ~= nil)
        if (prop == "Off") then
            return
        end
        g_dbgTimer = C4:AddTimer(90, "MINUTES")
        dbgStatus("Debug Timer set to 90 Minutes")
        return
    end
    if (strProperty == "IR Code Cache") then
        gUseCache = (prop == "On")
    end
    if (strProperty == "Poll Contact Inputs (iTach/Flex)") then
        if (gStatePollTimer > 0) then gStatePollTimer = C4:KillTimer(gStatePollTimer) end
        if (prop == "On") then
            gStatePollTimer = C4:AddTimer(1, "SECONDS", true) -- repeating poll timer...
        end
    end
    if (strProperty == "Show Status") then
        if (prop == "Off") then
            C4:UpdateProperty("Status", "'Show Status' is off.")
        end
    end
    if (strProperty == "Invert Relay Pulse") then 
        gInvertRelayPulse = (prop == "Inverted")
    end
end


function dbg(strDebugText)
    if (g_dbgprint) then print(strDebugText) end
    if (g_dbglog) then C4:DebugLog("\r\n" .. strDebugText) end
end


function dbgStatus(strText)
    dbg(strText)
    if (Properties["Show Status"] ~= "Off") then
        gCurStar = gCurStar + 1
        if (STARS[gCurStar] == nil) then gCurStar = 1 end
        C4:UpdateProperty("Status", STARS[gCurStar] .. " " .. strText)
    end
end


function UpdateConnStatus()
    gConnectedSince = gConnectedSince or os.time()
    local diff = os.difftime(os.time(), gConnectedSince)
    -- diff is in Seconds... Convert to Days, Hours, Minutes, Seconds...
    local min = math.floor(diff / 60)
    diff = diff - (min * 60)
    local hour = math.floor(min / 60)
    min = min - (60 * hour)
    local day = math.floor(hour / 24)
    hour = hour - (day * 24)
    
    local stat = "Connected to Global Cache "
    local timeinfo = string.format("%d:%02d:%02d", hour, min, diff)
    local qinfo = "     --     Cmd Queue: " .. #gCmdQueue .. "     IR Queue: " .. #gIRQueue
    if (day < 1) then
        stat = stat .. timeinfo .. qinfo
    else
        if (day == 1) then
            stat = stat .. day .. " Day, " .. timeinfo .. qinfo
        else
            stat = stat .. day .. " Days, " .. timeinfo .. qinfo
        end
    end
    C4:UpdateProperty("Connection Status", stat)
    if (Properties["Show Status"] ~= "Off") then
        gCurStar = gCurStar + 1
        if (STARS[gCurStar] == nil) then gCurStar = 1 end
        C4:UpdateProperty("Status", STARS[gCurStar] .. " " .. "Connected to Global Cache.")
    end
end


function OnTimerExpired(idTimer)
    if (idTimer == g_dbgTimer) then
        dbgStatus("Turning Debug Mode Off (timer expired)")
        C4:UpdateProperty("Debug Mode", "Off")
        OnPropertyChanged("Debug Mode")
        g_dbgTimer = 0
    end
    
    if (idTimer == g_InIRTimer) then
        g_IN_IR = false
        SendNextIR()
        g_InIRTimer = C4:KillTimer(g_InIRTimer)
        return
    end
    
    if (idTimer == gPollTimer) then
        -- TODO: Remove test code line below...
        --C4:SendToNetwork(gGCBinding, 4999, "blah")
        -- Send a status query on the main port...
        -- If we haven't gotten a response, increment the 'not checked in'... if it hits 3, disconnect all ports...
        gLastCheckin = gLastCheckin or 0
        gLastCheckin = gLastCheckin + 1
        if (gLastCheckin > 2) then
            gLastCheckin = 0
            if (gNetworkStatus == "ONLINE") then
                C4:NetDisconnect(gGCBinding, gGCControlPort)
                dbgStatus("Failed to receive poll responses... Disconnecting...")
                C4:ErrorLog("Global Cache device is not responding... Disconnecting...")
            end
            for k,v in pairs(gSerialOnline) do
                if (v == "ONLINE") then C4:NetDisconnect(gGCBinding, k) end
            end
        end
        -- Send Poll Packet if we're ONLINE...
        if (gNetworkStatus == "ONLINE") then
            UpdateConnStatus()
            SendGCCommand("getversion", "GetVersion (poll)")
        end
        return
    end
    
    if (idTimer == gStatePollTimer) then
        -- For each connection that's a CONTACT, poll it's current state...
        GetRelayContactSettings("Get Contact State (Poll)", true)
        return
    end
    
    if (idTimer == gReconnectTimer) then
        dbgStatus("Attempting to reconnect to Global Cache...")
        C4:ErrorLog("Attempting to reconnect to Global Cache...")
        local bTrying = false
        gReconnectTimer = C4:KillTimer(gReconnectTimer)
        if (gNetworkStatus == "OFFLINE") then
            -- Try to reconnect to the GC Control port...
            C4:NetDisconnect(gGCBinding, gGCControlPort)
            C4:NetConnect(gGCBinding, gGCControlPort)
            bTrying = true
        end
        for k,v in pairs(gSerialOnline) do
            if (v == "OFFLINE") then
                C4:NetDisconnect(gGCBinding, k)
                C4:NetConnect(gGCBinding, k)
                bTrying = true
                return
            end
        end
        if (bTrying) then
            gReconnectTimer = C4:AddTimer(5, "SECONDS")
        end
    end
    
    if (idTimer == gIRDelayTimer) then
        SendNextIR()
    end
    
    if (idTimer == gCmdTimer) then
        SendNextCmd()
    end
    
    if (idTimer == gTriggerTimer) then
        if (gNextCmd ~= nil) then
            SendGCCommand(gNextCmd, "Set Relay State")
        end
        gNextCmd = nil
    end
    
    if (idTimer == gSerialSettingsTimer) then
        -- Get Serial Settings on init, so we can do reconnect of serial on reboot...
        C4:CreateNetworkConnection(gDirectorBinding, "127.0.0.1", gDirectorPort)
        GetSerialSettings()
    end
    
    if (idTimer == g_DeviceIterationTimer) then
        -- Timed out on waiting for device list, go ahead and create the bindings...
        CreateBindings()
    end
    
    -- Kill Stray Timers...
    C4:KillTimer(idTimer)
end


function SendNextCmd()
    if (gCmdTimer ~= nil) and (gCmdTimer ~= 0) then gCmdTimer = C4:KillTimer(gCmdTimer) end
    local cmdtable = table.remove(gCmdQueue, 1)
    if (cmdtable ~= nil) then
        local cmd = cmdtable.CMD
        gCmdTimer = C4:AddTimer(100, "MILLISECONDS")
        --    if (cmdtable.NAME ~= "GetVersion (poll)") then
        --      dbgStatus("Sending: " .. cmdtable.NAME)
        --    end
        if (string.find(cmd, "sendir") == 1) then
            g_IN_IR = true
            g_InIRTimer = C4:AddTimer(10, "SECONDS")  -- Drop out of 'in IR' after 10 seconds if no completeir was found...
        end
        SendToGC(cmd)
    end
end


function SendGCCommand(strCmd, strName)
    gCmdQueue = gCmdQueue or {}
    table.insert(gCmdQueue, {CMD = strCmd, NAME = strName})
    if (gCmdTimer == nil) or (gCmdTimer == 0) then
        SendNextCmd()
    end
end


function AddDevice(nAddress, nCount, strType)
    dbg("Adding: " .. nAddress .. " (" .. nCount .. "): " .. strType)
    local temp = {}
    temp.Address = nAddress
    temp.Count = nCount
    temp.Type = strType
    temp.Bound = "F"
    table.insert(gDevices, temp)
end


-- Creates Control bindings after all binding info has been retrieved from the device...
function CreateBindings()
    dbg("Creating Bindings...")
    gGetDevices = false
    PersistData.ControlBindings = {}
    ID_TO_ADDRESS, ADDRESS_TO_ID = {}, {}
    local idBinding = 1
    local TYPE_ID = {}
    TYPE_ID["IR"] = 1
    TYPE_ID["IR_BLASTER"] = 1
    TYPE_ID["FLEX_IR"] = 1
    TYPE_ID["FLEX_IR_BLASTER"] = 1
    TYPE_ID["IRTRIPORT"] = 1
    TYPE_ID["IRTRIPORT_BLASTER"] = 1
    TYPE_ID["SERIAL"] = 1
    TYPE_ID["FLEXSERIAL"] = 1
    TYPE_ID["RELAY"] = 1
    TYPE_ID["RELAYSENSOR"] = 1
    TYPE_ID["SENSOR"] = 1
    TYPE_ID["SENSOR_NOTIFY"] = 1
    local SerPort = FIRST_SERIAL_IP_PORT
    for k,v in pairs(gDevices) do
        for i = 1, v.Count do
            for _, type in pairs(BINDING_TYPES[v.Type]) do
                local temp = {}
                temp.ID = idBinding
                temp.Name = TYPE_NAME[v.Type] .. TYPE_ID[v.Type]
                temp.Type = type
                temp.Address = v.Address .. ":" .. i
                if (type == "RS_232") then
                    temp.Port = SerPort
                end
                if (type == "RS_485") then
                    temp.Port = SerPort
                    SerPort = SerPort + 1
                end
                
                C4:AddDynamicBinding(temp.ID, "CONTROL", true, temp.Name, temp.Type, false, false)
                temp[v.Type] = TYPE_ID[v.Type]
                ADDRESS_TO_ID[temp.Address] = temp.ID
                ID_TO_ADDRESS[temp.ID] = temp.Address
                PersistData.ControlBindings[idBinding] = temp
                idBinding = idBinding + 1
            end
            TYPE_ID[v.Type] = TYPE_ID[v.Type] + 1
        end
    end
    
    -- Ensure that the PersistData gets saved, so we have it for bootup restoring...
    C4:InvalidateState()
end


-- Removes all Control bindings found in PersistData.ControlBindings
function DeleteBindings()
    for k,v in pairs(PersistData.ControlBindings) do
        C4:RemoveDynamicBinding(v.ID)
    end
    PersistData.ControlBindings = {}
    gDevices = {}
end


-- When we get a serial binding connected, we need to connect to the appropriate port if we're network bound.
-- Mark the serial bindings as bound when we get the change...
function OnBindingChanged(idBinding, strClass, bIsBound)
    if (bIsBound) then
        dbg("OnBindingChanged[" .. idBinding .. " -- " .. strClass .. "]: bound")
    else
        dbg("OnBindingChanged[" .. idBinding .. " -- " .. strClass .. "]: unbound")
    end
    
    local cb = PersistData.ControlBindings[idBinding]
    if (strClass == "RS_232" or strClass == "RS_485") then
        if (gNetworkBound) then
            local port = cb.Port
            C4:NetDisconnect(gGCBinding, port)
            if (bIsBound) then
                GetSerialSettings()  -- Setup serial port(s) when we have a serial binding...
                dbg("Connecting to serial connection " .. port)
                if (gSerialOnline[port] == nil) or (gSerialOnline[port] == "OFFLINE") then
                    C4:NetDisconnect(gGCBinding, port)
                    C4:NetConnect(gGCBinding, port)
                end
                cb.Bound = "T"
            end
        end
    end
    
    if (strClass == "IR_OUT") then
        -- Need to set output to IR mode, not Contact mode...
        local irmode = cb.IRMODE or "IR"
        if (not isFlex()) then
            SendGCCommand("set_IR," .. cb.Address .. "," .. irmode, "Set IR Mode")
        end
        
        -- Clear Code Cache for this binding...
        gCodeCache = gCodeCache or {}
        gCodeCache[idBinding] = {}
    end
    
    if (strClass == "CONTACT_SENSOR") then
        -- Need to set output to Contact mode, not IR mode...
        if (not isFlex()) then
            SendGCCommand("set_IR," .. cb.Address .. ",SENSOR_NOTIFY", "Set Contact Mode")  -- *not* "SENSOR"
        end
        SendGCCommand("getstate," .. cb.Address, "Get State")
        
        if (bIsBound) then
            cb.Bound = "T"
            -- If iTach, set the Poll Timer...
            gStatePollTimer = gStatePollTimer or 0
            C4:KillTimer(gStatePollTimer)
            if (Properties["Poll Contact Inputs (iTach/Flex)"] ~= "Off") then
                gStatePollTimer = C4:AddTimer(1, "SECONDS", true) -- repeating poll timer...
            end
        else
            cb.Bound = "F"
            -- See if any are bound.  If all are not bound, kill timer...
            local anybound = false
            for k,v in pairs(PersistData.ControlBindings) do
                if ((v.Type == "CONTACT_SENSOR") or (v.Type == "RELAY")) and (v.Bound == "T") then
                    anybound = true
                end
            end
            if (anybound == false) then gStatePollTimer = C4:KillTimer(gStatePollTimer) end
        end
        
    end
    
    -- Defect #36142 -- Global Cache relay not showing feedback -- Bound not getting set for relays...
    if (strClass == "RELAY") then
        if (bIsBound) then
            cb.Bound = "T"
            -- If iTach, set the Poll Timer...
            gStatePollTimer = gStatePollTimer or 0
            C4:KillTimer(gStatePollTimer)
            if (Properties["Poll Contact Inputs (iTach/Flex)"] ~= "Off") then
                gStatePollTimer = C4:AddTimer(1, "SECONDS", true) -- repeating poll timer...
            end
        else
            cb.Bound = "F"
            -- See if any are bound.  If all are not bound, kill timer...
            local anybound = false
            for k,v in pairs(PersistData.ControlBindings) do
                if ((v.Type == "CONTACT_SENSOR") or (v.Type == "RELAY")) and (v.Bound == "T") then
                    anybound = true
                end
            end
            if (anybound == false) then gStatePollTimer = C4:KillTimer(gStatePollTimer) end
        end
    end
end

function isGlobalConnect()
    return string.match(Version, "4001") -- IR
        or string.match(Version, "4002") -- SL
        or string.match(Version, "4003") -- RO
     -- or string.match(Version, "4004") -- SW
end

function isFlex()
    return string.match(Version, "2000") or string.match(Version, "3000")
end

function isGC100()
    return string.match(Version, "3.0-")
end

function OnNetworkBindingChanged(idBinding, bIsBound)
    if (idBinding == gGCBinding) then
        gNetworkBound = bIsBound
    end
end


function OnConnectionStatusChanged(idBinding, nPort, strStatus)
    dbg("OnConnectionStatusChanged[" .. idBinding .. " (" .. nPort .. ")]: " .. strStatus .. " -- " .. os.date())
    if (idBinding == gDirectorBinding) then
        if (strStatus == "ONLINE") then
            -- Send first serial settings retrieval step...
            dbg("Director Binding ONLINE... Sending...")
            C4:SendToNetwork(gDirectorBinding, gDirectorPort, '<c4soap name="GetBindingsByDevice" async="False"><param name="iddevice" type="number">' .. C4:GetDeviceID() .. '</param></c4soap>' .. string.char(0))
        end
        return
    end
    
    if (idBinding == gGCBinding) then
        if (nPort == gGCControlPort) then
            gNetworkStatus = strStatus
            if (strStatus == "ONLINE") then
                dbgStatus("Connected to Global Cache.")
                dbg("Global Cache Connection Up: " .. os.date() .. "-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-")
                gConnectedSince = gConnectedSince or os.time()
                if (gPollTimer ~= 0) then C4:KillTimer(gPollTimer) end
                gPollTimer = C4:AddTimer(5, "SECONDS", true) -- 5 second repeating timer...
                -- If we don't have device info, get it...
                if (PersistData.ControlBindings == nil) then
                    gGetDevices = true
                    SendGCCommand("getdevices", "Get Devices")
                else
                    -- If we already have bindings for this device, see if anything is connected to them, and get their serial settings...
                    GetSerialSettings()
                    GetRelayContactSettings()
                end
            else
                -- Dump queues if disconnected from device...
                dbg("Global Cache Connection Down: " .. os.date() .. "-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-=-")
                gConnectedSince = nil
                g_IN_IR = false
                gIRQueue = {}
                
                -- Try a re-connect of the device gGCControlPort...
                if (gReconnectTimer == nil) or (gReconnectTimer == 0) then
                    g_IN_IR = false
                    gIRQueue = {}
                    gReconnectTimer = C4:AddTimer(5, "SECONDS")
                    C4:NetDisconnect(gGCBinding, gGCControlPort)
                    C4:NetConnect(gGCBinding, gGCControlPort)
                end
            end
        else
            -- Mark serial port online/offline status...
            gSerialOnline[nPort] = strStatus
            if (strStatus == "OFFLINE") then
                -- Serial Port OFFLINE
                -- *DO NOT* try an immediate reconnect, as it will continually fail and retry if the device goes away or changes IP, this will slow director.
                if (gReconnectTimer == nil) or (gReconnectTimer == 0) then
                    gReconnectTimer = C4:AddTimer(5, "SECONDS")
                    C4:NetDisconnect(gGCBinding, nPort)
                    C4:NetConnect(gGCBinding, nPort)
                end
            end
        end
    end
end


function GetSerialSettings()
    dbg("GetSerialSettings called...")
    gSerialSettings = {}
    gDirectorBuf = ""
    C4:NetDisconnect(gDirectorBinding, gDirectorPort)
    C4:NetConnect(gDirectorBinding, gDirectorPort, "TCP")
end


function SendToGC(cmd)
    dbg("--------> " .. cmd)
    C4:SendToNetwork(gGCBinding, gGCControlPort, cmd .. string.char(0x0d))
end


function GetRelayContactSettings(strName, bNoQueue)
    strName = strName or "Get State"
    bNoQueue = bNoQueue or false
    -- Send GC Command to retrieve each relay and contact setting...
    -- Note: We check for state even of addresses configured as IR, not Contact...
    for k,v in pairs(PersistData.ControlBindings) do
        if ((v.Type == "CONTACT_SENSOR") or (v.Type == "RELAY")) and (v.Bound == "T") then
            if (bNoQueue) then
                SendToGC("getstate," .. v.Address)
            else
                SendGCCommand("getstate," .. v.Address, strName)
            end
        end
    end
end


function PulseIR(addr, code, name, count, offset)
    count = count or 1
    offset = offset or 1
    
    local mycode = code.code
    if (code.hasalt) then
        if (code.usealt) then
            mycode = code.alt
        end
        code.usealt = not code.usealt
    end
    local out = "sendir," .. addr .. ",1," .. code.freq .. "," .. count * code.repeatcount .. "," .. offset .. "," .. mycode
    if (string.len(out) > 300) then
        out = CompressIR(out)
    end
    
    if (count == 1) then
        SendGCCommand(out, "Pulse IR: " .. name)
    else
        SendGCCommand(out, "Start IR: " .. name)
    end
end

function CompressIR(code)
    local output = ""
    local pulses = {}
    local countdown = 6
    local currentKey = "A"
    local pulse = "";
    
    for part in string.gmatch(code, '([^,]+)') do
        if (countdown > 0) then
            if (countdown == 6) then
                output = part
            else
                output = output .. "," .. part 
            end
            countdown = countdown - 1
            
        elseif (countdown == 0) then 
            countdown = -1
            pulse = part
            
        else
            countdown = 0
            pulse = pulse .. "," .. part
            if (pulses[pulse] == nil) then
                if (currentKey <= "O") then
                    pulses[pulse] = currentKey
                    currentKey = string.char(string.byte(currentKey) + 1)
                end
                
                if string.find(output, "[A-O]$") then
                    output = output .. pulse
                else 
                    output = output .. "," .. pulse
                end
            else
                output = output .. pulses[pulse]
            end
        end
    end
    
    return output
end


function StartIR(addr, code, name)
    PulseIR(addr, code, name, tonumber(Properties["Max IR Repeats"]), 1)
end


function StopIR(addr)
    SendToGC("stopir," .. addr)
    --  dbgStatus("Sending: Stop IR")
    g_IN_IR = false
    SendNextIR()
end


function ConvertIR(code)
    local codes = {}
    local paircount = 0
    
    -- Convert from code string (Control4 Code format) to GC On/Off Format...
    code = string.gsub(code, " ", string.char(0), 5000)
    local pos, _, freq, pairs1, pairs2 = string.unpack(code, "zzzz")
    freq = tonumber(freq, 16)
    pairs1 = tonumber(pairs1, 16)
    pairs2 = tonumber(pairs2, 16)
    freq = math.ceil(1000000 / (freq * .241246))
    dbg("pos: " .. pos .. " freq: " .. freq .. " pairs1: " .. pairs1 .. " pairs2: " .. pairs2)
    paircount = pairs1 + pairs2
    
    local pos, val1, val2 = string.unpack(code, "zz", pos)
    while (val1 ~= nil) and (val2 ~= nil) do
        table.insert(codes, tonumber(val1, 16))
        table.insert(codes, tonumber(val2, 16))
        pos, val1, val2 = string.unpack(code, "zz", pos)
    end
    
    return table.concat(codes, ","), freq
end


function ReceivedFromProxy(idBinding, strCommand, tParams)
    --  if (strCommand ~= "PULSE") and (strCommand ~= "START") and (strCommand ~= "STOP") then
    dbg("ReceivedFromProxy[" .. idBinding .. "]: " .. strCommand)
    if (tParams ~= nil) then for k,v in pairs(tParams) do dbg(k .. ": " .. v) end end
    --  end
    
    if (PersistData.ControlBindings == nil) then dbgStatus("Control bindings not found!") return end
    local cb = PersistData.ControlBindings[idBinding]
    if (cb == nil) then dbgStatus("Control bindings not found for idBinding: " .. idBinding .. "!") return end
    
    -- CONTACT_SENSOR means IR as well...
    if (cb.Type == "CONTACT_SENSOR" or cb.Type == "IR_OUT") then
        if (strCommand == "STOP") then
            --      dbg("STOP IR received.")
            StopIR(cb.Address)
            
            -- If there are START commands for this binding are in the IR queue, we NEED TO REMOVE THEM...
            for k,v in pairs(gIRQueue) do
                if (v.BINDING == idBinding) then
                    gIRQueue[k] = nil
                end
            end
            
            -- If commands for this port are found in the outgoing command queue, we NEED TO REMOVE THEM...
            for k,v in pairs(gCmdQueue) do
                if (v.CMD ~= nil) then
                    if (string.find(v.CMD, cb.Address) ~= nil) then
                        gCmdQueue[k] = nil
                    end
                end
            end
            return
        end
        
        if (strCommand == "PULSE") or (strCommand == "START") then
            HandleIR(idBinding, strCommand, tParams.data)
            return
        end
    end
    
    if (cb.Type == "RS_232" or cb.Type == "RS_485") then
        if (strCommand == "SEND") then
            if (cb.Bound == "T") then
                dbgStatus("Sending Serial -> Port: " .. cb.Port)
                dbg("--------> (Port " .. cb.Port .. ") " .. tParams["data"])
                C4:SendToNetwork(gGCBinding, cb.Port, C4:Base64Decode(tParams["data"]))
            end
        end
        return
    end
    
    if (cb.Type == "RELAY") then
        if (gNetworkStatus == "OFFLINE") then
            return		-- If we're offline, the relay won't get activated, so we're sunk...
        end
        
        if (strCommand == "GET_STATE") then
            -- Send command to get state of this relay...
            SendGCCommand("getstate," .. cb.Address, "Get Relay / Contact State")
            return
        end
        if (strCommand == "OPEN") then
            cb.State = "OPENED"
        end
        if (strCommand == "CLOSE") then
            cb.State = "CLOSED"
        end
        if (strCommand == "TOGGLE") then
            if (cb.State == "OPENED") then
                cb.State = "CLOSED"
            else
                cb.State = "OPENED"
            end
        end
        
        -- TODO: If there is a way to tell whether this TRIGGER should be inverted, that may be a better solution.
        if (strCommand == "TRIGGER") then
            if (gInvertRelayPulse) then
                cb.State = "OPENED"
                gNextCmd = "setstate," .. cb.Address .."," .. RELAYSTATE["CLOSED"]
            else
                cb.State = "CLOSED"
                gNextCmd = "setstate," .. cb.Address .."," .. RELAYSTATE["OPENED"]
            end
            
            local time = tonumber(tParams.TIME) or 500
            gTriggerTimer = C4:AddTimer(time, "MILLISECONDS")
        end
        SendGCCommand("setstate," .. cb.Address .."," .. RELAYSTATE[cb.State], "Set Relay State")
        return
    end
end


function HandleIR(idBinding, strCommand, data)
    -- Convert CODE to something the GC can accept...
    gCodeCache = gCodeCache or {}
    gCodeCache[idBinding] = gCodeCache[idBinding] or {}
    
    if ((strCommand == "PULSE") or (strCommand == "START")) then
        local _, _, CMD = string.find(data, "<name>(.-)</name>")
        if (string.find(data, "<codes") == 1) then
            -- Multiple codes in a macro...  Put them in the IR Macro Queue...
            local inp = string.gsub(data, "<codes>", "")
            inp = string.gsub(inp,"</codes>", "")
            local first, last, innerxml
            last = 1
            local _, _, key = string.find(inp, "<(%w-)%W", last)
            while (key ~= nil) do
                first, last, innerxml = string.find(inp, "<" .. key .. ".->(.-)</" .. key .. ">", last)
                dbg("Key: " .. key .. " First: " .. first .. " Last: " .. last .. " innerxml: " .. innerxml)
                
                -- Parse individual code if code... parse delay if delay...  Put each into the gIRMacro queue...
                if (key == "delay") then
                    table.insert(gIRMacroQueue, {BINDING = idBinding, COMMAND = "DELAY", CODE = innerxml, NAME = ""})
                end
                
                if (key == "ircode") then
                    -- Find CMD name, it's not the outer command...
                    _, _, CMD = string.find(innerxml, "<name>(.-)</name>")
                    local newcode = nil
                    if (gUseCache) then newcode = gCodeCache[idBinding][CMD] end
                    if (newcode == nil) then
                        newcode = decodeC4IR(CMD, innerxml)
                        if (gUseCache) then gCodeCache[idBinding][CMD] = newcode end
                    end
                    -- newcode now has the converted code... Put it into the general cache and process if not processing.
                    table.insert(gIRMacroQueue, {BINDING = idBinding, COMMAND = strCommand, CODE = newcode, NAME = CMD})
                    if (newcode.delayafter > 0) then
                        table.insert(gIRMacroQueue, {BINDING = idBinding, COMMAND = "DELAY", CODE = newcode.delayafter, NAME = ""})
                    end
                    SendNextIR()
                end
                
                _, _, key = string.find(inp, "<(%w-)%W", last)
            end
            
        elseif (string.find(data, "<ircode") == 1) then
            
            -- Dump it into the General queue, process if it's alone in the queue...
            local newcode = nil
            if (gUseCache) then newcode = gCodeCache[idBinding][CMD] end
            if (newcode == nil) then
                newcode = decodeC4IR(CMD, data)
                if (gUseCache) then gCodeCache[idBinding][CMD] = newcode end
            end
            -- newcode now has the converted code... Put it into the general cache and process if not processing.
            table.insert(gIRQueue, {BINDING = idBinding, COMMAND = strCommand, CODE = newcode, NAME = CMD})
            if (newcode.delayafter > 0) then
                table.insert(gIRQueue, {BINDING = idBinding, COMMAND = "DELAY", CODE = newcode.delayafter, NAME = ""})
            end
            SendNextIR()
        end
    end
end


function decodeC4IR(cmdname, data)
    cmdname = cmdname or "No Name"
    dbg("Decoding IR...")
    local code = {}
    local _, _, CODE = string.find(data, "<pattern>(.-)</pattern>")
    local _, _, ALT = string.find(data, "<altpattern>(.-)</altpattern>")
    if (ALT == nil) then ALT = "" end
    local _, _, REPEAT = string.find(data, "<repeatcount>(.-)</repeatcount>")
    local _, _, DELAYAFTER = string.find(data, "<delayafter>(.-)</delayafter>")
    CODE = string.gsub(CODE, "%s+$", "")
    ALT = string.gsub(ALT, "%s+$", "")
    dbg("Cmd: [" .. cmdname .. "] Code: [" .. CODE .. "] Alt: [" .. ALT .. "]")
    code.repeatcount = tonumber(REPEAT)
    code.delayafter = tonumber(DELAYAFTER) or 0
    code.usealt = false		-- Whether to send ALT or normal...
    code.hasalt = false
    code.code, code.freq = ConvertIR(CODE)
    if (ALT ~= "") then
        code.alt = ConvertIR(ALT)
        code.hasalt = true
    end
    return code
end


function SendNextIR()
    if (g_IN_IR) then return end
    
    if (gNetworkStatus ~= "ONLINE") then
        g_IN_IR = false
        gIRQueue = {}
        return
    end
    
    local fullcmd = table.remove(gIRQueue, 1)
    if (fullcmd ~= nil) then
        local cb = PersistData.ControlBindings[fullcmd.BINDING]
        
        if (fullcmd.COMMAND == "PULSE") then
            dbg("Pulsing " .. fullcmd.NAME)
            PulseIR(cb.Address, fullcmd.CODE, fullcmd.NAME)
            return
        end
        
        if (fullcmd.COMMAND == "START") then
            dbg("Starting " .. fullcmd.NAME)
            StartIR(cb.Address, fullcmd.CODE, fullcmd.NAME)
            return
        end
        
        if (fullcmd.COMMAND == "DELAY") then
            dbgStatus("Delay: " .. fullcmd.CODE .. "MS")
            gIRDelayTimer = C4:AddTimer(fullcmd.CODE, "MILLISECONDS")
            SendNextCmd()  -- If we're starting a delay timer, go ahead and send the next (non-IR) command...
            return
        end
        
    else
        -- No commands currently in queue... Check the Macro queue...
        if (#gIRMacroQueue > 0) then
            dbg("Sending next Macro Item...")
            table.insert(gIRQueue, table.remove(gIRMacroQueue, 1))
            SendNextIR()
        else
            SendNextCmd()
        end
    end
end


function string:split(sep)
    local sep, fields = sep or ":", {}
    local pattern = string.format("([^%s]+)", sep)
    self:gsub(pattern, function(c) fields[#fields+1] = c end)
    return fields
end


function HandleDirectorMsg(strData)
    -- Serial Settings retrieval...
    gDirectorBuf = gDirectorBuf .. strData
    -- Check for \000 in buffer, parse reply if \000 received...
    if (string.find(gDirectorBuf, string.char(0)) ~= nil) then
        if (string.find(gDirectorBuf, "GetBindingsByDevice") ~= nil) then
            dbg("GetBindingsByDevice found.")
            -- For each <binding>, look for *2* Device ID's...  And index by the *first* binding id...
            
            for binding in string.gfind(gDirectorBuf, "<binding>(.-)</binding>") do
                if (string.find(binding, "RS_232") ~= nil) then
                    local _, _, bindingid = string.find(binding, "<bindingid>(.-)</bindingid>")
                    local _, _, bound = string.find(binding, "<bound>(.-)</bound>")
                    if (bound ~= nil) then
                        local _, _, deviceid = string.find(bound, "<deviceid>(.-)</deviceid>")
                        dbg("Binding " .. bindingid .. " is bound to Device ID: " .. deviceid)
                        gSerialSettings[bindingid] = {}
                        gSerialSettings[bindingid].deviceid = deviceid
                        gSerialSettings[bindingid].serialsettings = ""
                        gSerialSettings[bindingid].c4i = ""
                    end
                end
            end
            
            gDirectorBuf = ""
            GetNextSerialSetting()
        end
        if (string.find(gDirectorBuf, "GetItem") ~= nil) then
            dbg("GetItem found.")
            local _, _, deviceid = string.find(gDirectorBuf, "<id>(.-)</id>")
            local _, _, c4i_name = string.find(gDirectorBuf, "<config_data_file>(.-)</config_data_file>")
            for k,v in pairs(gSerialSettings) do if (v.deviceid == deviceid) then v.c4i = c4i_name end end
            gDirectorBuf = ""
            C4:SendToNetwork(gDirectorBinding, gDirectorPort, '<c4soap name="GetProjectC4i" async="False"><param name="name" type="string">' .. c4i_name .. '</param></c4soap>' .. string.char(0))
        end
        if (string.find(gDirectorBuf, "GetProjectC4i") ~= nil) then
            dbg("GetProjectC4i found.")
            -- Parse out the <serialsettings> section, set it on the GC unit...
            local _, _, serialsettings = string.find(gDirectorBuf, "<serialsettings>(.-)</serialsettings>")
            for k,v in pairs(gSerialSettings) do
                if (v.c4i ~= "") and (v.serialsettings == "") then
                    serialsettings = string.gsub(serialsettings, "\n", " ") .. " "
                    v.serialsettings = serialsettings
                    local bindingid = tonumber(k)
                    local splitstring = serialsettings:split(" ")
                    local baud = splitstring[1] or "9600"
                    local bits = splitstring[2] or "8"
                    local parity = splitstring[3] or "none"
                    local stop = splitstring[4] or "1"
                    local flow = splitstring[5] or "none"
                    dbg("Binding: " .. bindingid .. " DeviceID: " .. v.deviceid .. " Serialsettings: " .. v.serialsettings .. " -- Baud: [" .. baud .."] Bits: [" .. bits .. "] Parity: [" .. parity .. "] Stop: [" .. stop .. "] Flow: [" .. flow .. "]")
                    local gcc = "set_SERIAL," .. ID_TO_ADDRESS[bindingid] .. "," .. baud .. "," .. FLOW[flow] .. "," .. PARITY[parity]
                    SendGCCommand(gcc, "Set Serial Settings")
                    local port = PersistData.ControlBindings[bindingid].Port
                    if (gSerialOnline[port] == nil) or (gSerialOnline[port] == "OFFLINE") then
                        PersistData.ControlBindings[bindingid].Bound = "T"
                        C4:NetDisconnect(gGCBinding, port)
                        C4:NetConnect(gGCBinding, port)
                    end
                end
            end
            gDirectorBuf = ""
            
            GetNextSerialSetting()
        end
    end
end


function ReceivedFromNetwork(idBinding, nPort, strData)
    -- Stuff from serial ports...
    if (idBinding == gGCBinding) then
        dbg("<-------- " .. strData) 
        if (strData:find("version,") or strData:find("%d%d%d%-%d%d%d%d%-%d")) then
            Version = strData
        end
        if (nPort == gGCControlPort) then
            local endpos = string.find(strData, string.char(0x0d))
            while (endpos ~= nil) do
                -- Get out the message, process it.
                local msg = string.sub(strData, 1, endpos - 1)
                
                -- Remove it
                strData = string.sub(strData, endpos+1)
                
                -- Process packet (uses pcall for protection)
                local ret, err = pcall(HandleMsg, msg)
                if (ret ~= true) then
                    local e = "Error Calling HandleMsg: " .. err
                    dbgStatus(e)
                    C4:ErrorLog(e)
                end
                
                -- Check for the next one...
                endpos = string.find(strData, string.char(0x0d))
            end
        else
            -- Find which serial binding this data should go to...
            for k,v in pairs(PersistData.ControlBindings) do
                if (v.Port == nPort) then
                    C4:SendToProxy(k, "SERIAL_DATA", C4:Base64Encode(strData), "NOTIFY")
                    dbgStatus("Received Serial <- Port: " .. nPort)
                end
            end
            return
        end
    end
    
    if (idBinding == gDirectorBinding) then
        -- Process packet (uses pcall for protection)
        local ret, err = pcall(HandleDirectorMsg, strData)
        if (ret ~= true) then
            local e = "Error Calling HandleDirectorMsg: " .. err
            dbgStatus(e)
            C4:ErrorLog(e)
        end
    end
end



function GetNextSerialSetting()
    -- Get next serial item from gSerialSettings table...
    for k,v in pairs(gSerialSettings) do
        if (v.serialsettings == "") then
            C4:SendToNetwork(gDirectorBinding, gDirectorPort, '<c4soap name="GetItem" async="False"><param name="iditem" type="number">' .. v.deviceid .. '</param><param name="filter" type="number"></param></c4soap>' .. string.char(0))
            return
        end
    end
    dbg("Found all serial settings... Closing Director connection")
    C4:NetDisconnect(gDirectorBinding, gDirectorPort)
end


function ExecuteCommand(strCommand, tParams)
    --dbg("ExecuteCommand: " .. strCommand)
    --if (tParams ~= nil) then for k,v in pairs(tParams) do dbg(k .. " " .. v) end end
    if (strCommand == "Blink") then
        SendGCCommand("blink," .. BLINK[tParams["Blink"]], "Blink")
    end
    if (strCommand == "Set IR Mode") then
        -- Set specified IR output to No Carrier / IR with Carrier, based on param...
        if (not isGC100()) then 
            --This command is only really valid for GC-100 units.
            return
        end
        
        local out = tonumber(tParams["IR Output"])
        local mode = tParams["IR Mode"]
        
        for k,v in pairs(PersistData.ControlBindings) do
            if (v.IR == out) then
                SendGCCommand("set_IR," .. v.Address .. "," .. IRMODE[mode], "IR Mode")
                PersistData.ControlBindings[k].IRMODE = IRMODE[mode]
                C4:InvalidateState()
            end
        end
    end
    if (strCommand == "LUA_ACTION") then
        if tParams ~= nil then
            if (tParams.ACTION == "RECREATE_BINDINGS") then
                DeleteBindings()
                gDevices = {}
                gGetDevices = true
                SendGCCommand("getdevices", "Get Devices")
            end
            if (tParams.ACTION == "PRINT_PORTINFO") then
                local out = {}
                PersistData = PersistData or {}
                PersistData.ControlBindings = PersistData.ControlBindings or {}
                for k,v in pairs(PersistData.ControlBindings) do table.insert(out, string.rep("-", 40)) for k1, v1 in pairs(v) do if (type(v1) ~= "boolean") then table.insert(out, k1 .. ": " .. v1) end end end
                table.insert(out, string.rep("-", 40))
                print(table.concat(out, "\n"))
            end
            if (tParams.ACTION == "PRINT_CONNECTIONS") then
                local out = {}
                table.insert(out, "--------------------------- Connected GC Ports ---------------------------")
                gNetworkStatus = gNetworkStatus or "No Status"
                gSerialOnline = gSerialOnline or {}
                table.insert(out, "4998\t" .. gNetworkStatus)
                for k,v in pairs(gSerialOnline) do table.insert(out, k .. "\t" .. v) end
                print(table.concat(out, "\n"))
            end
        end
    end
end


function HandleMsg(strMsg)
    --  if (string.find(strMsg, "ersio ") == nil) then dbg("HandleMsg: " .. strMsg) end
    gLastCheckin = 0
    
    -- Remove line termination (0x0D)
    strMsg = string.gsub(strMsg, string.char(0x0d), "")
    
    -- If looking to add devices, check for 'device.' and 'endlistdevices' messages...
    if (gGetDevices) then
        if (string.find(strMsg, "device.") == 1) then
            -- TODO -- Figure out why this errors on editor, and if replacement works...
            --local _, _, address, count, devtype = string.find(strMsg, "device\.(.+)\.(.+) (.+)")
            local _, _, address, count, devtype = string.find(strMsg, "device.(.+).(.+) (.+)")
            dbg("Adding Device: Address " .. address .. " Count: " .. count .. " Devtype: " .. devtype)
            
            if (isFlex()) then
                if (devtype == "SERIAL") then
                    devtype = "FLEXSERIAL"
                elseif (devtype == "IR") then
                    devtype = "FLEX_IR"
                elseif (devtype == "IR_BLASTER") then
                    devtype = "FLEX_IR_BLASTER"
                end
            elseif (isGlobalConnect()) then
                if (devtype == "IR_OUT") then
                    devtype = "IR"
                elseif (devtype == "SENSOR_DIGITAL") then
                    address = 0
                    count = 0
                elseif (devtype == "SERIAL_RS232") then
                    devtype = "SERIAL"
                elseif (devtype == "RELAY_SPST_3A") then
                    devtype = "RELAY"
                end
            end
            
            if (devtype == "IRTRIPORT_BLASTER") then
                AddDevice(address,"2","FLEX_IR")
                AddDevice(address,"1","FLEX_IR_BLASTER")
            elseif (devtype == "IRTRIPORT") then 
                dbg("Adding Triport")
                AddDevice(address,"3","FLEX_IR")
            elseif (devtype == "SENSOR_NOTIFY" or devtype == "SENSOR") then
                AddDevice(address,"4",devtype)
            elseif (devtype == "RELAYSENSOR") then
                AddDevice(address,"4","RELAY")
                AddDevice(address+1,"4","SENSOR")
                
            else
                AddDevice(tonumber(address), tonumber(count), devtype)
            end
            
            if (g_DeviceIterationTimer ~= 0) then 
                C4:KillTimer(g_DeviceIterationTimer) 
            end
            
            g_DeviceIterationTimer = C4:AddTimer(8, "SECONDS")
            return
        end
        if (string.find(strMsg, "endlistdevices") == 1) then
            if (g_DeviceIterationTimer ~= 0) then C4:KillTimer(g_DeviceIterationTimer) end
            CreateBindings()
            SendNextCmd()
            return
        end
    end
    
    -- Relay state change returned on setstate or getstate, need to notify proxy of new state.
    if (string.find(strMsg, "state,") == 1) then
        local _, _, addr, val = string.find(strMsg, "state,(.+),(.+)")
        gLastState = gLastState or {}
        gLastState[addr] = gLastState[addr] or ""
        if (gLastState[addr] ~= val) then
            local id = ADDRESS_TO_ID[addr]
            local cb = PersistData.ControlBindings[id]
            cb.State = RELAYSTATE[val]
            C4:SendToProxy(ADDRESS_TO_ID[addr], cb.State, {}, "NOTIFY")
            dbgStatus("Received: Relay / Contact " .. addr .. " State Change")
        end
        gLastState[addr] = val
        return
    end
    
    -- State Change of contact, need to notify contact proxy of new state...
    if (string.find(strMsg, "statechange,") == 1) then
        local _, _, addr, val = string.find(strMsg, "statechange,(.+),(.+)")
        gLastState = gLastState or {}
        gLastState[addr] = gLastState[addr] or ""
        if (gLastState[addr] ~= val) then
            C4:SendToProxy(ADDRESS_TO_ID[addr], RELAYSTATE[val], {}, "NOTIFY")
            dbgStatus("Received: Relay / Contact " .. addr .. " State Change")
        end
        gLastState[addr] = val
        return
    end
    
    -- When we complete an IR, we should check for additional IR on this port's macro queue...
    -- If none found, check all other port's macro queues...
    if (string.find(strMsg, "completeir,") == 1) then
        --    dbgStatus("Received: Complete IR")
        g_IN_IR = false
        SendNextIR()
    end
    
    SendNextCmd()
    
end

print("Driver Loaded..." .. os.date())



-- -=-=-=-=-=-=-=-=-=-=-=-  INIT  -=-=-=-=-=-=-=-=-=-=-=-
g_dbgprint, g_dbglog, g_dbgTimer = 0, 0, 0
gStatePollTimer, gPollTimer, g_DeviceIterationTimer = 0, 0, 0

-- Initialize all properties with their first changed values...

for k,v in pairs(Properties) do
    OnPropertyChanged(k)
end

gNetworkBound = false
gGetDevices = false
gDevices = {}
gSerialOnline = {}
gCmdQueue = {}

-- Restore Bindings on driver load, if binding info was saved...
PersistData = PersistData or {}
gIRMacroQueue = {}
gIRQueue = {}  -- Global queue, used so only an individual command will be sent at a time, because of the GC-100 limitation...

ID_TO_ADDRESS, ADDRESS_TO_ID = {}, {}
if (PersistData.ControlBindings ~= nil) then
    for k,v in pairs(PersistData.ControlBindings) do
        -- CONTACT_SENSOR bindings also need the IR binding (they're 2-in-one on the same binding ID, IR first...)
        if (v.Type == "CONTACT_SENSOR") then
            C4:AddDynamicBinding(v.ID, "CONTROL", true, v.Name, "IR_OUT", false, false)
        end
        C4:AddDynamicBinding(v.ID, "CONTROL", true, v.Name, v.Type, false, false)
        ID_TO_ADDRESS[v.ID] = v.Address
        ADDRESS_TO_ID[v.Address] = v.ID
        PersistData.ControlBindings[k].Bound = "F"  -- Set all to false initially, when serial bindings happen, we need to connect...
    end
end

gSerialSettingsTimer = C4:AddTimer(5, "SECONDS")
dbgStatus("Driver Initialized")

